/*
*********************************************************************************************************
** I2C master mode should be set in main application
*********************************************************************************************************
*/


#include  "i2cint.h" 
#include "lpc23xx.h"
#include "irq.h"
#include "target.h"

/* I2C Interface*/
#define I2CONSET      I20CONSET
#define I2STAT        I20STAT
#define I2DAT         I20DAT
#define I2ADR         I20ADR
#define I2SCLH        I20SCLH
#define I2SCLL        I20SCLL
#define I2CONCLR      I20CONCLR

#define I2CONSET_I2EN		0x00000040  /* I2C Control Set Register */
#define I2CONSET_AA			0x00000004
#define I2CONSET_SI			0x00000008
#define I2CONSET_STO		0x00000010
#define I2CONSET_STA		0x00000020

#define I2CONCLR_AAC		0x00000004  /* I2C Control clear Register */
#define I2CONCLR_SIC		0x00000008
#define I2CONCLR_STAC		0x00000020
#define I2CONCLR_I2ENC		0x00000040


/* Global variables handled in interrupt */
volatile UNS_8 	I2C_sla;			/* I2C slave address	*/
volatile UNS_32	I2C_suba;			/* I2C chip internal sub address */
volatile UNS_8 	I2C_suba_num;		/* num of bytes of I2C sub address	*/
volatile UNS_8 	*I2C_buf;        	/* pointer to data buffer	*/
volatile UNS_32 I2C_num;			/* num of data to be write/read	*/
volatile UNS_8 	I2C_end;			/* flag for I2C bus operation finish, set to 1 when finished	*/
volatile UNS_8 	I2C_suba_en;		/* 	sub address control
										0--sub address has been handled or do not need sub address
										1--read
										2--write
									*/
									
/****************************************************************************
* Name: 	Wait_I2c_End
* Function:	software delay, wait the I2C operation to finish
* Input:	dly delay count
* Output:	TRUE if I2C bus operation finished before time out, otherwise return FALSE
****************************************************************************/
UNS_8  Wait_I2c_End(UNS_32  dly)
{  UNS_32  i;

   if( I2C_end==1 ) return (TRUE);
   for(; dly>0; dly--) 
      for(i=0; i<5000; i++)
      {
        if( I2C_end==1 ) return (TRUE);
      }
   return (FALSE);   
}

/*
*********************************************************************************************************
** Name:	I2C_ReadNByte()
** Function:	Read N bytes from I2C slave with sub address
** Input:		sla			I2C slave address
**				suba_type	sub address type	1-one byte 	28+X	2two bytes
**				suba		sub address
**				s			pinter to data receive buffer
**				num			num to read
** Output:		TRUE		success
**				FALSE		failure
*********************************************************************************************************
*/
UNS_8 I2C_ReadNByte (UNS_8 sla, UNS_32 suba_type, UNS_32 suba, UNS_8 *s, UNS_32 num)
{
	if (num <= 0 )
		return FALSE;
		
	/* config parameters */
	if (suba_type == ONE_BYTE_SUBA)
	{	/* sigle byte sub address */
		I2C_sla     	= sla + 1;							/* read, R=1 	*/
		I2C_suba    	= suba;								/* sub address of slave	*/
		I2C_suba_num	= 1;								/* number of bytes of sub address */
	}
	if (suba_type == TWO_BYTE_SUBA)
	{	/* double bytes sub address */
		I2C_sla     	= sla + 1;							/* read, R=1 	*/
		I2C_suba   	 	= suba;								/* sub address of slave	*/
		I2C_suba_num	= 2;								/* number of bytes of sub address */
	}
	if (suba_type == X_ADD_8_SUBA)
	{	/* sub address type: 8+X*/
		I2C_sla			= sla + ((suba >> 7 )& 0x0e) + 1;	/* read, R=1 	*/
		I2C_suba		= suba & 0x0ff;						/* sub address of slave	*/
		I2C_suba_num	= 1;								
	}
	I2C_buf     = s;										/* pointer to receive buffer */
	I2C_num     = num;										/* num of bytes to read*/
	I2C_suba_en = 1;										/* need to read the sub address	*/
	I2C_end     = 0;
	
	/* clear STA,SI,AA */
	I2CONCLR = 	(1 << 2)|	/* AA 		*/
				(1 << 3)|	/* SI 		*/
				(1 << 5);	/* STA 		*/
	
	/* setSTA,start I2C bus */
	I2CONSET = 	(1 << 5)|	/* STA 		*/
				(1 << 6);	/* I2CEN 	*/
	
	/* wait the I2C operation to complete */
	return( Wait_I2c_End(20));
	
	/*while (I2C_end == 0)
	{
	 
	}
	if (I2C_end == 1)
		return (TRUE);
	else
		return (FALSE);			*/
	
}

/*
*********************************************************************************************************
** Name:	I2C_WriteNByte()
** Function:	Write N bytes to I2C slave with sub address
** Input:		sla			slave address
**				suba_type	sub address type	1-one byte 	28+X	2two bytes
**			  	suba		sub address
**			  	*s			pointer to buffer which contain data to write
**			  	num			num of write
** Output:		TRUE		success
**			  	FALSE		failure
*********************************************************************************************************
*/
UNS_8 I2C_WriteNByte(UNS_8 sla, UNS_8 suba_type, UNS_32 suba, UNS_8 *s, UNS_32 num)
{
	if (num <= 0 )
		return FALSE;
		

	/* config parameter*/	
	if (suba_type == ONE_BYTE_SUBA)
	{	/* sub address: one byte */
		I2C_sla     	= sla;								
		I2C_suba    	= suba;								
		I2C_suba_num	= 1;							
	}
	if (suba_type == TWO_BYTE_SUBA)
	{	/* sub address: two bytes */
		I2C_sla     	= sla;								
		I2C_suba   	 	= suba;								
		I2C_suba_num	= 2;								
	}
	if (suba_type == X_ADD_8_SUBA)
	{	/* sub address:8+X */
		I2C_sla			= sla + ((suba >> 7 )& 0x0e);		
		I2C_suba		= suba & 0x0ff;						
		I2C_suba_num	= 1;								
	}

	I2C_buf     = s;										/* data				*/
	I2C_num     = num;										/* number 			*/
	I2C_suba_en = 2;										/* has sub address, write */
	I2C_end     = 0;
	
	/* clear STA,SI,AA */
	I2CONCLR = 	(1 << 2)|	/* AA 	*/
				(1 << 3)|	/* SI 	*/
				(1 << 5);	/* STA 	*/
	
	/* set STA,start I2C operation */
	I2CONSET = 	(1 << 5)|	/* STA 	*/
				(1 << 6);	/* I2CEN*/
	
	/* wait the I2C to complete */
    return( Wait_I2c_End(20));
    		/*while (I2C_end == 0)
	{
	 // 
	}
	if (I2C_end == 1)
		return (TRUE);
	else
		return (FALSE);			*/

}

/*
*********************************************************************************************************
** Name:	__irq IRQ_I2C()
** Function:	isr for I2C 
** Input:	no
** Output:	no 
*********************************************************************************************************
*/
void __irq IRQ_I2C(void)
{	/* read I2DAT */
	/* operate and set flag based on the global variables */
	/* clar interrupt flag, return */
	
	switch (I2STAT & 0xF8)
	{	
		case 0x08:	/* A Start condition is issued. */
			/* load SLA+W or SLA+R */
		 	if(I2C_suba_en == 1)/* SLA+R */				/* specify the adress to read */
		 	{	I2DAT = I2C_sla & 0xFE; 				/* write address first	*/
		 	}
            else	/* SLA+W */
            {  	I2DAT = I2C_sla;        				/* send slave address */
            }
            /* clar SI */
            I2CONCLR =	(1 << 3)|						/* SI 						*/
            			(1 << 5);						/* STA 						*/
            break;
            
       	case 0x10:	/* A repeated started is issued */ 	
       		/* load SLA+W or SLA+R */
       		I2DAT = I2C_sla;							/* re-send slaver address after restart I2C */
       		I2CONCLR = 0x28;							/* clear SI,STA */
       		break;

		case 0x18:	
       	case 0x28:	/* Data byte has been transmitted, ACK received */
       		if (I2C_suba_en == 0)
       		{
	       		if (I2C_num > 0)
	       		{	I2DAT = *I2C_buf++;
	       			I2CONCLR = 0x28;					/* clear SI,STA 				*/
	       			I2C_num--;
	       		}
	       		else	/* no more data to send */
	       		{		/* stop bus*/
	       		  	I2CONSET = (1 << 4);				/* STO 						*/
	       			I2CONCLR = 0x28;					/* clearSI,STA 				*/
	       		  	I2C_end = 1;						/* bus stopped 			*/
	       		}
       		}
       		
            if(I2C_suba_en == 1)	/* re-start the I2C if read with sub address	*/
            { 
            	if (I2C_suba_num == 2)
            	{	I2DAT = ((I2C_suba >> 8) & 0xff);
	       			I2CONCLR = 0x28;					/* clear SI,STA 				*/
	       			I2C_suba_num--;
	       			break;	
	       		} 
	       		
	       		if(I2C_suba_num == 1)
	       		{	I2DAT = (I2C_suba & 0xff);
	       			I2CONCLR = 0x28;					/* clear SI,STA 				*/
	       			I2C_suba_num--;
	       			break;	
	       		}
	       		
            	if (I2C_suba_num == 0)
            	{	I2CONSET = 0x20;
               		I2CONCLR = 0x08;
               		I2C_suba_en = 0;     				/* sub address is handled	*/
               		break;
               	}
            }
            
            if (I2C_suba_en == 2)
       		{
       		 	if (I2C_suba_num > 0)
            	{	if (I2C_suba_num == 2)
            		{	I2DAT = ((I2C_suba >> 8) & 0xff);
            			I2CONCLR = 0x28;
            			I2C_suba_num--;
            			break;
            		}
            		if (I2C_suba_num == 1)
            		{	I2DAT    = (I2C_suba & 0xff);
               			I2CONCLR = 0x28;
               			I2C_suba_num--;
               			I2C_suba_en  = 0;
               			break;
               		}
               	}
             }
       		break;
       		  
       case 0x40:	/* Master Receive, SLA_R has been sent */
       		if (I2C_num <= 1)	/* last byte */			
       		{	I2CONCLR = 1 << 2;      				/* send NACK next time 		*/
       		}
       		else
       		{ 	I2CONSET = 1 << 2;						/* send ACK next time	*/
       		}
       		I2CONCLR = 0x28;							/* clear SI,STA 				*/
       		break;

       	case 0x20:	/* regardless, it's a NACK */
       	case 0x30:	/* data in I2DAT has been sent, already received NACK */
       	case 0x38:	
   		case 0x48:	/* SLA+R has been sent,received NACK */
         	I2CONCLR = 0x28;
            I2C_end = 0xFF; 
       		break;   				
	
		case 0x50:	/* Data byte has been received, regardless following ACK or NACK */
			*I2C_buf++ = I2DAT;
			I2C_num--;
			if (I2C_num == 1)/* last byte */
			{  	I2CONCLR = 0x2C;						/* STA,SI,AA = 0 			*/
			}
			else
			{  	I2CONSET = 0x04;						/* AA=1 					*/
			  	I2CONCLR = 0x28;
			}
			break;
		
		case 0x58:	/* received byte, NACK has been sent back */
			*I2C_buf++ = I2DAT;     					/* read last byte*/
            I2CONSET = 0x10;        					/* stop I2C bus*/
            I2CONCLR = 0x28;
            I2C_end = 1; 
            break;
            
      	default:
      		break;
	}
   VICVectAddr = 0x00;              					
}

/****************************************************************************
* Name:	I2C_Init
* Function:	Initialize I2C as master mode
* Input:	fi2c		I2C bus speed, should be less than 400kbps
* Output:	no
* Note: the SDA bus requires a pullup resistor to VCC (typical 10 k for 100 kHz, 
* 2 k for 400 kHz)
****************************************************************************/
UNS_8 I2C_Init( UNS_32 fi2c ) 
{
	if(fi2c>400000) fi2c = 400000;
	
  PCONP |= (1 << 19);
  PINSEL1 &= ~0x03C00000;
  PINSEL1 |= 0x01400000;	/* set PIO0.27 and PIO0.28 to I2C0 SDA and SCK */
							/* function to 01 on both SDA and SCK. */
  /*--- Clear flags ---*/
  I2CONCLR = I2CONCLR_AAC | I2CONCLR_SIC | I2CONCLR_STAC | I2CONCLR_I2ENC;    
	//I2CONCLR = 0x2C;
  /*--- Reset registers ---*/
  I2SCLL   = (Fpclk/fi2c) / 2;
  I2SCLH   = (Fpclk/fi2c + 1) / 2;
    
  /* Install interrupt handler */	
  if ( install_irq( I2C0_INT, (void *)IRQ_I2C, LOWEST_PRIORITY/*HIGHEST_PRIORITY */) == FALSE )
  {
	return( FALSE );
  }
  I2CONSET = I2CONSET_I2EN;		
  return( TRUE );
}
